/*!
 *  \~chinese
 *  @header     PTBitmap.h
 *  @abstract   图片处理
 *
 *  \~english
 *  @header     PTBitmap.h
 *  @abstract   Image manager
 */

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>
#import <CoreGraphics/CoreGraphics.h>
#import "SDKDefine.h"


/*!
 *  \~chinese
 *  压缩模式
 *
 *  \~english
 *  Compress Mode
 */
typedef NS_ENUM(NSInteger,PTBitmapCompressMode) {
    
    PTBitmapCompressModeNone = 0,   /*! *\~chinese 不压缩 *\~english None */
    PTBitmapCompressModeZPL2 = 16,  /*! *\~chinese ZPL2压缩算法 *\~english ZPL2 compress */
    PTBitmapCompressModeTIFF = 32,  /*! *\~chinese TIFF压缩算法 *\~english TIFF compress */
    PTBitmapCompressModeLZO = 48,   /*! *\~chinese LZO压缩算法 *\~english LZO compress */
};

/*!
 *  \~chinese
 *  位图模式
 *
 *  \~english
 *  Bitmap Mode
 */
typedef NS_ENUM(NSInteger, PTBitmapMode) {
    
    PTBitmapModeBinary = 0,     /*! *\~chinese 黑白二值图像 *\~english Binary */
    PTBitmapModeDithering = 1,  /*! *\~chinese 灰阶抖动图像 *\~english Dithering */
    PTBitmapModeColumn = 2,     /*! *\~chinese 无效 *\~english not supported */
};

/*!
 *  \~chinese
 *  哪种指令处理图片
 *
 *  \~english
 *  Which instruction handles the image
 */
typedef NS_ENUM(NSInteger, PTBitmapCommand) {
    
    PTBitmapCommandTSPL = 0,    /*! *\~chinese TSPL指令 *\~english TSPL command */
    PTBitmapCommandOther = 1,   /*! *\~chinese 其他的指令 *\~english Other command */
};


@interface PTBitmap : NSObject


/*!
*  \~chinese
*  数据取反处理，eg：TSPL的位图处理需要取反
*
*  @param data  输入数据
*  @return      取反输出数据
*
*  \~english
*  Data negation processing
*
*  @param data  input data
*  @return      data negation
*/
+ (NSData *)negativeData:(NSData *)data;


/*!
 *  \~chinese
 *  生成打印机打印图片数据
 *
 *  @param image    图片
 *  @param mode     生成的位图数据类型 简单的黑白二值化或者抖动处理
 *  @param compress 数据压缩类型
 *  @return         提供给打印机使用的图片数据
 *
 *  \~english
 *  Generate data of printing image for the printer
 *
 *  @param image    image
 *  @param mode     Type of generated bitmap data; simple black and white image or dithering
 *  @param compress Type of data compression
 *  @return         Image data provided to the printer
 */
+ (NSData *)getImageData:(CGImageRef)image mode:(PTBitmapMode)mode compress:(PTBitmapCompressMode)compress;

/*!
 *  \~chinese
 *  生成打印机打印图片数据
 *
 *  @param image    图片
 *  @param mode     生成的位图数据类型 简单的黑白二值化或者抖动处理
 *  @param compress 数据压缩类型
 *  @param command  指令类型
 *  @return         提供给打印机使用的图片数据
 *
 *  \~english
 *  Generate data of printing image for the printer
 *
 *  @param image    image
 *  @param mode     Type of generated bitmap data; simple black and white image or dithering
 *  @param compress Type of data compression
 *  @param command  Type of command
 *  @return         Image data provided to the printer
 */
+ (NSData *)getImageData:(CGImageRef)image mode:(PTBitmapMode)mode compress:(PTBitmapCompressMode)compress command:(PTBitmapCommand)command;

/*!
 *  \~chinese
 *  用column算法生成的图片数据
 *
 *  @param sourceBitmap  输入数据
 *  @return             位图数据
 *
 *  \~english
 *  Image data generated by the column algorithm
 *
 *  @param sourceBitmap  input data
 *  @return              bitmap Data
 */
+ (NSData *)generateColumnData:(CGImageRef)sourceBitmap;
    
    
/*!
 *  \~chinese
 *  将bitmap数据转成图片
 *
 *  @param image    图片
 *  @param mode     生成的位图数据类型 简单的黑白二值化或者抖动处理
 *  @return         预览的图片
 *
 *  \~english
 *  Convert bitmap data to image
 *
 *  @param image    image
 *  @param mode     Type of generated bitmap data; simple black and white image or dithering
 *  @return         Review Image
 */
+ (UIImage *)generateRenderingWithImage:(CGImageRef)image mode:(PTBitmapMode)mode;
    
@end




