

import UIKit
import Photos

extension UIAlertController {
    
    static func showActionAlert(_ title:String, message:String, confirm:String, cancelHandle:((UIAlertAction) -> Void)?,confirmHandle:((UIAlertAction) -> Void)?) {
        
        guard let rootVC = AppWindow.rootViewController else { return }
        
        let alert = UIAlertController.init(title: title, message: message, preferredStyle: .alert)
        let cancel = UIAlertAction.init(title: NSLocalizedString("Cancel", comment: ""), style: .cancel, handler: cancelHandle)
        let confirm = UIAlertAction.init(title: confirm, style: .default, handler: confirmHandle)
        alert.addAction(cancel)
        alert.addAction(confirm)
        
        alert.popoverPresentationController?.sourceView = rootVC.view
        alert.popoverPresentationController?.sourceRect = CGRect.init(x: 0, y: rootVC.view.bounds.height - 1, width: rootVC.view.bounds.width, height: 1)
        
        rootVC.present(alert, animated: true, completion: nil)
    }
    
    static func showActionAlert(_ title:String, message:String, confirm:String, confirmHandle:((UIAlertAction) -> Void)?) {
        
        guard let rootVC = AppWindow.rootViewController else { return }
        
        let alert = UIAlertController.init(title: title, message: message, preferredStyle: .alert)
        let cancel = UIAlertAction.init(title: NSLocalizedString("Cancel", comment: ""), style: .cancel, handler: nil)
        let confirm = UIAlertAction.init(title: confirm, style: .default, handler: confirmHandle)
        alert.addAction(cancel)
        alert.addAction(confirm)
        
        alert.popoverPresentationController?.sourceView = rootVC.view
        alert.popoverPresentationController?.sourceRect = CGRect.init(x: 0, y: rootVC.view.bounds.height - 1, width: rootVC.view.bounds.width, height: 1)
        
        rootVC.present(alert, animated: true, completion: nil)
    }
}

struct PTAssetCollection {
    
    var collection : PHAssetCollection
    
    var fetchResult : PHFetchResult<PHAsset>
    
    init(with collection:PHAssetCollection, fetchResult:PHFetchResult<PHAsset>) {
        self.collection = collection
        self.fetchResult = fetchResult
    }
    
}

typealias PTImageParameterClosure = ((_: UIImage) -> Void)

class PTPhotoListViewController: UIViewController {

    var selectClosure : PTImageParameterClosure?
    
    var smalltAlbums : PHFetchResult<PHAssetCollection>!
    var userAlbums : PHFetchResult<PHCollection>!
    
    var collectionArray = [PTAssetCollection]()
    
    lazy var tableView : UITableView = {
        let temp = UITableView.init(frame:self.view.bounds, style: .grouped)
        temp.autoresizingMask = [.flexibleWidth,.flexibleHeight]
        temp.backgroundColor = .white
        temp.delegate = self
        temp.dataSource = self
        temp.estimatedRowHeight = 0
        temp.estimatedSectionFooterHeight = 0
        temp.estimatedSectionHeaderHeight = 0
        temp.separatorInset = .zero
        temp.register(UINib.init(nibName: "LWPhotosListCell", bundle: nil), forCellReuseIdentifier: "iden")
        self.view.addSubview(temp)
        return temp
    }()
    
    override func viewDidLoad() {
        super.viewDidLoad()

        view.backgroundColor = UIColor.white
        navigationItem.title = NSLocalizedString("Album list", comment: "")
        determinePermission()
    }

    func determinePermission() -> Void {
        
        let photoLibrary = PHPhotoLibrary.authorizationStatus()
        switch photoLibrary {
        case .denied,.restricted:
            
            UIAlertController.showActionAlert(NSLocalizedString("Tips", comment: ""), message: NSLocalizedString("Please set to allow the application to access your camera \n set > privacy > camera", comment: ""), confirm: "OK") { (_) in
                let url = URL.init(string: UIApplication.openSettingsURLString)
                UIApplication.shared.openURL(url!)
            }
        case .notDetermined:
            PHPhotoLibrary.requestAuthorization({ (status) in
                if status == PHAuthorizationStatus.authorized {
                    DispatchQueue.main.async {
                        self.getPhotoList()
                    }
                }
            })
        
        case .authorized:
            DispatchQueue.main.async {
                self.getPhotoList()
            }
        }
        
        
    }

    func getPhotoList() -> Void {
        let fetchOptions = PHFetchOptions.init()
        fetchOptions.sortDescriptors = [NSSortDescriptor.init(key: "creationDate", ascending: false)]
        
        //获取全部系统相册
        smalltAlbums = PHAssetCollection.fetchAssetCollections(with: .smartAlbum, subtype: .albumRegular, options: nil)
        //获取全部用户相册
        userAlbums = PHAssetCollection.fetchTopLevelUserCollections(with: nil)
        
        smalltAlbums.enumerateObjects ({ (collection, index, finished) in

            switch collection.assetCollectionSubtype {
            case .smartAlbumVideos:
                break
            default:
                let result = PHAsset.fetchAssets(in: collection, options: fetchOptions)
                if result.count != 0 {
                    let temp = PTAssetCollection.init(with: collection, fetchResult: result)
                    self.collectionArray.append(temp)
                }
            }
        })

        userAlbums.enumerateObjects ({ (collection, index, finished) in
            if let wapCollection = collection as? PHAssetCollection {
                let result = PHAsset.fetchAssets(in: wapCollection, options: fetchOptions)
                if result.count != 0 {
                    let temp = PTAssetCollection.init(with: wapCollection, fetchResult: result)
                    self.collectionArray.append(temp)
                }
            }
        })
        
        self.collectionArray.sort { (collection1, collection2) -> Bool in
            return collection1.fetchResult.count > collection2.fetchResult.count
        }
        
        self.tableView.reloadData()
    }
}


extension PTPhotoListViewController : UITableViewDelegate,UITableViewDataSource {
    
    func tableView(_ tableView: UITableView, heightForHeaderInSection section: Int) -> CGFloat {
        
        return 0.01
    }
    
    func tableView(_ tableView: UITableView, viewForHeaderInSection section: Int) -> UIView? {
        
        return nil
    }
    
    func tableView(_ tableView: UITableView, heightForFooterInSection section: Int) -> CGFloat {
        
        return 0.01
    }
    
    func tableView(_ tableView: UITableView, viewForFooterInSection section: Int) -> UIView? {
        
        return nil
    }
    
    func numberOfSections(in tableView: UITableView) -> Int {
        
        return 1
    }
    
    func tableView(_ tableView: UITableView, numberOfRowsInSection section: Int) -> Int {
        
        return collectionArray.count
    }
    
    func tableView(_ tableView: UITableView, cellForRowAt indexPath: IndexPath) -> UITableViewCell {
        let cell = tableView.dequeueReusableCell(withIdentifier: "iden") as! LWPhotosListCell
        if let localTitle = collectionArray[indexPath.row].collection.localizedTitle {
            cell.collectionName.text = localTitle
        }
        cell.collectionCount.text = "\(collectionArray[indexPath.row].fetchResult.count)"
        
        if let asset = collectionArray[indexPath.row].fetchResult.firstObject {
            cell.collectionImage.loadImage(with: asset, targetSize: CGSize.init(width: 120, height: 120))
        }
        
        return cell
    }
    
    func tableView(_ tableView: UITableView, heightForRowAt indexPath: IndexPath) -> CGFloat {
        return 80
    }
    
    func tableView(_ tableView: UITableView, didSelectRowAt indexPath: IndexPath) {
        
        tableView.deselectRow(at: indexPath, animated: true)
        let result = collectionArray[indexPath.row].fetchResult
        
        let photo = PTAssetViewController.init(with: .image, assets: result)
        photo.assetClosure = selectClosure
        navigationController?.pushViewController(photo, animated: true)
        
    }
}



